<?php

namespace App\Console\Commands;

use App\Models\Product;
use App\Notifications\LowStockAlert;
use App\Models\User;
use Illuminate\Console\Command;

class CheckLowStockCommand extends Command
{
    protected $signature = 'stock:check-low
                          {--notify : Send notifications to admins}';

    protected $description = 'Check for low stock products';

    public function handle(): int
    {
        $this->info('Checking for low stock products...');

        $lowStockProducts = Product::with('inventory')
            ->get()
            ->filter(function ($product) {
                $totalStock = $product->inventory->sum('quantity');
                return $totalStock <= $product->min_stock;
            });

        if ($lowStockProducts->isEmpty()) {
            $this->info('No low stock products found.');
            return Command::SUCCESS;
        }

        $this->table(
            ['ID', 'Name', 'Current Stock', 'Min Stock'],
            $lowStockProducts->map(function ($product) {
                return [
                    $product->id,
                    $product->name,
                    $product->inventory->sum('quantity'),
                    $product->min_stock,
                ];
            })
        );

        if ($this->option('notify')) {
            $admins = User::where('role', 'admin')->get();
            foreach ($lowStockProducts as $product) {
                $currentStock = $product->inventory->sum('quantity');
                foreach ($admins as $admin) {
                    $admin->notify(new LowStockAlert($product, $currentStock));
                }
            }
            $this->info('Notifications sent to admins.');
        }

        return Command::SUCCESS;
    }
}
