<?php

namespace App\Console\Commands;

use App\Models\SalesOrder;
use App\Models\PurchaseOrder;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class GenerateDailyReportCommand extends Command
{
    protected $signature = 'report:generate-daily
                          {--date= : Specific date (Y-m-d format)}';

    protected $description = 'Generate daily sales and purchase report';

    public function handle(): int
    {
        $date = $this->option('date') ? 
                \Carbon\Carbon::parse($this->option('date')) : 
                now()->subDay();

        $this->info("Generating report for: {$date->toDateString()}");

        $sales = SalesOrder::whereDate('order_date', $date)
            ->where('status', 'completed')
            ->get();

        $purchases = PurchaseOrder::whereDate('order_date', $date)
            ->where('status', 'completed')
            ->get();

        $salesTotal = $sales->sum('total');
        $purchasesTotal = $purchases->sum('total');
        $profit = $salesTotal - $purchasesTotal;

        $reportData = [
            'date' => $date->toDateString(),
            'sales_count' => $sales->count(),
            'sales_total' => $salesTotal,
            'purchases_count' => $purchases->count(),
            'purchases_total' => $purchasesTotal,
            'profit' => $profit,
        ];

        Log::channel('daily')->info('Daily Report', $reportData);

        $this->table(
            ['Metric', 'Value'],
            [
                ['Sales Orders', $reportData['sales_count']],
                ['Sales Total', number_format($salesTotal, 0, ',', '.')],
                ['Purchase Orders', $reportData['purchases_count']],
                ['Purchases Total', number_format($purchasesTotal, 0, ',', '.')],
                ['Profit', number_format($profit, 0, ',', '.')],
            ]
        );

        $this->info('Report generated successfully!');

        return Command::SUCCESS;
    }
}
