<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Customer;
use Illuminate\Http\Request;

class CustomerController extends Controller
{
    public function index(Request $request)
    {
        $query = Customer::withCount(['salesOrders', 'debts']);

        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('phone', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            });
        }

        if ($request->has('has_debt') && $request->has_debt) {
            $query->where('total_debt', '>', 0);
        }

        $customers = $query->orderBy('created_at', 'desc')
                           ->paginate($request->per_page ?? 20);

        return response()->json($customers);
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'nullable|string|max:20',
            'email' => 'nullable|email|unique:customers,email',
            'address' => 'nullable|string',
        ]);

        $customer = Customer::create($request->all());

        return response()->json([
            'message' => 'Thêm khách hàng thành công',
            'customer' => $customer
        ], 201);
    }

    public function show($id)
    {
        $customer = Customer::with(['salesOrders.items.product', 'debts'])
                            ->findOrFail($id);

        return response()->json($customer);
    }

    public function update(Request $request, $id)
    {
        $customer = Customer::findOrFail($id);

        $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'nullable|string|max:20',
            'email' => 'nullable|email|unique:customers,email,' . $id,
            'address' => 'nullable|string',
        ]);

        $customer->update($request->all());

        return response()->json([
            'message' => 'Cập nhật khách hàng thành công',
            'customer' => $customer
        ]);
    }

    public function destroy($id)
    {
        $customer = Customer::findOrFail($id);

        if ($customer->total_debt > 0) {
            return response()->json([
                'message' => 'Không thể xóa khách hàng còn nợ'
            ], 422);
        }

        $customer->delete();

        return response()->json([
            'message' => 'Xóa khách hàng thành công'
        ]);
    }

    public function debts($id)
    {
        $customer = Customer::findOrFail($id);
        $debts = $customer->debts()
                         ->with(['salesOrder', 'payments'])
                         ->orderBy('created_at', 'desc')
                         ->get();

        return response()->json($debts);
    }
}
