<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\SalesOrder;
use App\Models\PurchaseOrder;
use App\Models\Customer;
use App\Models\Supplier;
use App\Models\Product;
use App\Models\CustomerDebt;
use App\Models\SupplierDebt;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    public function index(Request $request)
    {
        $fromDate = $request->from_date ?? now()->startOfMonth()->toDateString();
        $toDate = $request->to_date ?? now()->toDateString();

        // Revenue statistics
        $totalRevenue = SalesOrder::whereBetween('order_date', [$fromDate, $toDate])
                                  ->where('status', 'completed')
                                  ->sum('total');

        $totalPurchase = PurchaseOrder::whereBetween('order_date', [$fromDate, $toDate])
                                      ->where('status', 'completed')
                                      ->sum('total');

        $totalProfit = $totalRevenue - $totalPurchase;

        // Sales statistics
        $totalSalesOrders = SalesOrder::whereBetween('order_date', [$fromDate, $toDate])
                                      ->where('status', 'completed')
                                      ->count();

        $totalPurchaseOrders = PurchaseOrder::whereBetween('order_date', [$fromDate, $toDate])
                                            ->where('status', 'completed')
                                            ->count();

        // Customer statistics
        $totalCustomers = Customer::where('is_active', true)->count();
        $customersWithDebt = Customer::where('total_debt', '>', 0)->count();

        // Supplier statistics
        $totalSuppliers = Supplier::where('is_active', true)->count();
        $suppliersWithDebt = Supplier::where('total_debt', '>', 0)->count();

        // Debt statistics
        $customerDebts = CustomerDebt::where('status', '!=', 'paid')->sum('remaining_amount');
        $supplierDebts = SupplierDebt::where('status', '!=', 'paid')->sum('remaining_amount');

        // Product statistics
        $totalProducts = Product::where('is_active', true)->count();
        
        // Fix: Correct low stock query
        $lowStockProducts = Product::select('products.*')
            ->leftJoin('inventory', 'products.id', '=', 'inventory.product_id')
            ->where('products.is_active', true)
            ->groupBy('products.id', 'products.name', 'products.sku', 'products.category_id', 
                      'products.brand_id', 'products.unit_id', 'products.cost_price', 
                      'products.selling_price', 'products.min_stock', 'products.description', 
                      'products.is_active', 'products.created_at', 'products.updated_at',
                      'products.barcode')
            ->havingRaw('COALESCE(SUM(inventory.quantity), 0) <= products.min_stock')
            ->count();

        // Revenue by day (last 7 days)
        $revenueByDay = SalesOrder::select(
                DB::raw('DATE(order_date) as date'),
                DB::raw('SUM(total) as total')
            )
            ->where('order_date', '>=', now()->subDays(7))
            ->where('status', 'completed')
            ->groupBy('date')
            ->orderBy('date', 'asc')
            ->get();

        // Top selling products
        $topProducts = DB::table('sales_order_items')
            ->join('products', 'sales_order_items.product_id', '=', 'products.id')
            ->join('sales_orders', 'sales_order_items.sales_order_id', '=', 'sales_orders.id')
            ->whereBetween('sales_orders.order_date', [$fromDate, $toDate])
            ->where('sales_orders.status', 'completed')
            ->select(
                'products.id',
                'products.name',
                DB::raw('SUM(sales_order_items.quantity) as total_quantity'),
                DB::raw('SUM(sales_order_items.subtotal) as total_revenue')
            )
            ->groupBy('products.id', 'products.name')
            ->orderBy('total_revenue', 'desc')
            ->limit(10)
            ->get();

        return response()->json([
            'revenue' => [
                'total_revenue' => $totalRevenue,
                'total_purchase' => $totalPurchase,
                'total_profit' => $totalProfit,
            ],
            'orders' => [
                'total_sales_orders' => $totalSalesOrders,
                'total_purchase_orders' => $totalPurchaseOrders,
            ],
            'customers' => [
                'total' => $totalCustomers,
                'with_debt' => $customersWithDebt,
            ],
            'suppliers' => [
                'total' => $totalSuppliers,
                'with_debt' => $suppliersWithDebt,
            ],
            'debts' => [
                'customer_debts' => $customerDebts,
                'supplier_debts' => $supplierDebts,
                'net_debt' => $customerDebts - $supplierDebts,
            ],
            'products' => [
                'total' => $totalProducts,
                'low_stock' => $lowStockProducts,
            ],
            'revenue_by_day' => $revenueByDay,
            'top_products' => $topProducts,
        ]);
    }
}
