<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\CustomerDebt;
use App\Models\SupplierDebt;
use App\Models\DebtPayment;
use App\Models\Customer;
use App\Models\Supplier;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class DebtController extends Controller
{
    // Customer Debts
    public function customerDebts(Request $request)
    {
        $query = CustomerDebt::with(['customer', 'salesOrder', 'payments']);

        if ($request->has('customer_id')) {
            $query->where('customer_id', $request->customer_id);
        }

        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        if ($request->has('overdue') && $request->overdue) {
            $query->where('due_date', '<', now())
                  ->where('status', '!=', 'paid');
        }

        $debts = $query->orderBy('created_at', 'desc')
                       ->paginate($request->per_page ?? 20);

        return response()->json($debts);
    }

    // Supplier Debts
    public function supplierDebts(Request $request)
    {
        $query = SupplierDebt::with(['supplier', 'purchaseOrder', 'payments']);

        if ($request->has('supplier_id')) {
            $query->where('supplier_id', $request->supplier_id);
        }

        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        if ($request->has('overdue') && $request->overdue) {
            $query->where('due_date', '<', now())
                  ->where('status', '!=', 'paid');
        }

        $debts = $query->orderBy('created_at', 'desc')
                       ->paginate($request->per_page ?? 20);

        return response()->json($debts);
    }

    // Pay Customer Debt
    public function payCustomerDebt(Request $request, $id)
    {
        $request->validate([
            'amount' => 'required|numeric|min:0.01',
            'payment_date' => 'required|date',
            'payment_method' => 'required|in:cash,bank_transfer,card,other',
            'notes' => 'nullable|string',
        ]);

        DB::beginTransaction();

        try {
            $debt = CustomerDebt::findOrFail($id);

            if ($debt->status === 'paid') {
                return response()->json([
                    'message' => 'Công nợ đã được thanh toán đầy đủ'
                ], 422);
            }

            if ($request->amount > $debt->remaining_amount) {
                return response()->json([
                    'message' => 'Số tiền thanh toán lớn hơn số nợ còn lại'
                ], 422);
            }

            // Create payment record
            DebtPayment::create([
                'debt_type' => 'customer',
                'debt_id' => $debt->id,
                'amount' => $request->amount,
                'payment_date' => $request->payment_date,
                'payment_method' => $request->payment_method,
                'notes' => $request->notes,
                'user_id' => $request->user()->id,
            ]);

            // Update debt
            $debt->paid_amount += $request->amount;
            $debt->remaining_amount -= $request->amount;
            $debt->updateStatus();

            // Update customer total debt
            $customer = Customer::find($debt->customer_id);
            $customer->updateTotalDebt();

            DB::commit();

            return response()->json([
                'message' => 'Thanh toán công nợ thành công',
                'debt' => $debt->load(['customer', 'payments'])
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'message' => 'Lỗi: ' . $e->getMessage()
            ], 422);
        }
    }

    // Pay Supplier Debt
    public function paySupplierDebt(Request $request, $id)
    {
        $request->validate([
            'amount' => 'required|numeric|min:0.01',
            'payment_date' => 'required|date',
            'payment_method' => 'required|in:cash,bank_transfer,card,other',
            'notes' => 'nullable|string',
        ]);

        DB::beginTransaction();

        try {
            $debt = SupplierDebt::findOrFail($id);

            if ($debt->status === 'paid') {
                return response()->json([
                    'message' => 'Công nợ đã được thanh toán đầy đủ'
                ], 422);
            }

            if ($request->amount > $debt->remaining_amount) {
                return response()->json([
                    'message' => 'Số tiền thanh toán lớn hơn số nợ còn lại'
                ], 422);
            }

            // Create payment record
            DebtPayment::create([
                'debt_type' => 'supplier',
                'debt_id' => $debt->id,
                'amount' => $request->amount,
                'payment_date' => $request->payment_date,
                'payment_method' => $request->payment_method,
                'notes' => $request->notes,
                'user_id' => $request->user()->id,
            ]);

            // Update debt
            $debt->paid_amount += $request->amount;
            $debt->remaining_amount -= $request->amount;
            $debt->updateStatus();

            // Update supplier total debt
            $supplier = Supplier::find($debt->supplier_id);
            $supplier->updateTotalDebt();

            DB::commit();

            return response()->json([
                'message' => 'Thanh toán công nợ thành công',
                'debt' => $debt->load(['supplier', 'payments'])
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'message' => 'Lỗi: ' . $e->getMessage()
            ], 422);
        }
    }

    // Debt Summary
    public function summary()
    {
        $customerDebts = CustomerDebt::where('status', '!=', 'paid')->sum('remaining_amount');
        $supplierDebts = SupplierDebt::where('status', '!=', 'paid')->sum('remaining_amount');

        $overdueCustomerDebts = CustomerDebt::where('due_date', '<', now())
                                            ->where('status', '!=', 'paid')
                                            ->sum('remaining_amount');

        $overdueSupplierDebts = SupplierDebt::where('due_date', '<', now())
                                            ->where('status', '!=', 'paid')
                                            ->sum('remaining_amount');

        return response()->json([
            'customer_debts' => $customerDebts,
            'supplier_debts' => $supplierDebts,
            'overdue_customer_debts' => $overdueCustomerDebts,
            'overdue_supplier_debts' => $overdueSupplierDebts,
            'net_debt' => $customerDebts - $supplierDebts,
        ]);
    }
}
