<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderItem;
use App\Models\Inventory;
use App\Models\SupplierDebt;
use App\Models\Supplier;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class PurchaseOrderController extends Controller
{
    public function index(Request $request)
    {
        $query = PurchaseOrder::with(['supplier', 'warehouse', 'user', 'items.product']);

        if ($request->has('supplier_id')) {
            $query->where('supplier_id', $request->supplier_id);
        }

        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        if ($request->has('from_date')) {
            $query->whereDate('order_date', '>=', $request->from_date);
        }

        if ($request->has('to_date')) {
            $query->whereDate('order_date', '<=', $request->to_date);
        }

        $orders = $query->orderBy('order_date', 'desc')
                        ->paginate($request->per_page ?? 20);

        return response()->json($orders);
    }

    public function store(Request $request)
    {
        $request->validate([
            'supplier_id' => 'required|exists:suppliers,id',
            'warehouse_id' => 'required|exists:warehouses,id',
            'order_date' => 'required|date',
            'discount' => 'nullable|numeric|min:0',
            'paid_amount' => 'required|numeric|min:0',
            'notes' => 'nullable|string',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.unit_price' => 'required|numeric|min:0',
        ]);

        DB::beginTransaction();

        try {
            // Calculate totals
            $subtotal = 0;
            foreach ($request->items as $item) {
                $subtotal += $item['quantity'] * $item['unit_price'];
            }

            $discount = $request->discount ?? 0;
            $total = $subtotal - $discount;

            // Create purchase order (Observer will auto-generate order_number)
            $order = PurchaseOrder::create([
                'supplier_id' => $request->supplier_id,
                'warehouse_id' => $request->warehouse_id,
                'user_id' => $request->user()->id,
                'order_date' => $request->order_date,
                'subtotal' => $subtotal,
                'discount' => $discount,
                'total' => $total,
                'paid_amount' => $request->paid_amount,
                'payment_method' => $request->payment_method ?? 'cash',
                'status' => 'completed',
                'notes' => $request->notes,
            ]);

            // Create order items and update inventory
            foreach ($request->items as $item) {
                $itemSubtotal = $item['quantity'] * $item['unit_price'];

                PurchaseOrderItem::create([
                    'purchase_order_id' => $order->id,
                    'product_id' => $item['product_id'],
                    'quantity' => $item['quantity'],
                    'unit_price' => $item['unit_price'],
                    'subtotal' => $itemSubtotal,
                ]);

                // Update inventory (increase stock)
                $inventory = Inventory::firstOrCreate(
                    [
                        'warehouse_id' => $request->warehouse_id,
                        'product_id' => $item['product_id']
                    ],
                    ['quantity' => 0]
                );

                $inventory->quantity += $item['quantity'];
                $inventory->save();
            }

            // Create supplier debt if not fully paid
            $remainingAmount = $total - $request->paid_amount;
            if ($remainingAmount > 0) {
                SupplierDebt::create([
                    'supplier_id' => $request->supplier_id,
                    'purchase_order_id' => $order->id,
                    'amount' => $remainingAmount,
                    'paid_amount' => 0,
                    'remaining_amount' => $remainingAmount,
                    'status' => 'unpaid',
                ]);

                // Update supplier total debt
                $supplier = Supplier::find($request->supplier_id);
                $supplier->updateTotalDebt();
            }

            DB::commit();

            return response()->json([
                'message' => 'Tạo đơn mua hàng thành công',
                'order' => $order->load(['supplier', 'warehouse', 'items.product'])
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'message' => 'Lỗi: ' . $e->getMessage()
            ], 422);
        }
    }

    public function show($id)
    {
        $order = PurchaseOrder::with(['supplier', 'warehouse', 'user', 'items.product', 'debt'])
                              ->findOrFail($id);

        return response()->json($order);
    }

    public function cancel($id)
    {
        $order = PurchaseOrder::findOrFail($id);

        if ($order->status === 'cancelled') {
            return response()->json([
                'message' => 'Đơn hàng đã bị hủy trước đó'
            ], 422);
        }

        DB::beginTransaction();

        try {
            // Restore inventory (decrease stock)
            foreach ($order->items as $item) {
                $inventory = Inventory::where('warehouse_id', $order->warehouse_id)
                                     ->where('product_id', $item->product_id)
                                     ->first();

                if ($inventory) {
                    $inventory->quantity -= $item->quantity;
                    if ($inventory->quantity < 0) {
                        $inventory->quantity = 0;
                    }
                    $inventory->save();
                }
            }

            // Delete debt if exists
            if ($order->debt) {
                $order->debt->delete();
                $supplier = Supplier::find($order->supplier_id);
                $supplier->updateTotalDebt();
            }

            $order->status = 'cancelled';
            $order->save();

            DB::commit();

            return response()->json([
                'message' => 'Hủy đơn hàng thành công',
                'order' => $order
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'message' => 'Lỗi: ' . $e->getMessage()
            ], 422);
        }
    }
}
