<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\SalesOrder;
use App\Models\PurchaseOrder;
use App\Models\Inventory;
use App\Models\CustomerDebt;
use App\Models\SupplierDebt;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ReportController extends Controller
{
    // Revenue Report
    public function revenue(Request $request)
    {
        $fromDate = $request->from_date ?? now()->startOfMonth()->toDateString();
        $toDate = $request->to_date ?? now()->toDateString();

        $salesOrders = SalesOrder::with(['customer', 'items.product'])
                                 ->whereBetween('order_date', [$fromDate, $toDate])
                                 ->where('status', 'completed')
                                 ->orderBy('order_date', 'desc')
                                 ->get();

        $totalRevenue = $salesOrders->sum('total');
        $totalPaid = $salesOrders->sum('paid_amount');
        $totalDebt = $totalRevenue - $totalPaid;

        return response()->json([
            'summary' => [
                'total_revenue' => $totalRevenue,
                'total_paid' => $totalPaid,
                'total_debt' => $totalDebt,
                'total_orders' => $salesOrders->count(),
            ],
            'orders' => $salesOrders,
        ]);
    }

    // Inventory Report
    public function inventory(Request $request)
    {
        $query = Inventory::with(['warehouse', 'product.category', 'product.brand', 'product.unit']);

        if ($request->has('warehouse_id')) {
            $query->where('warehouse_id', $request->warehouse_id);
        }

        if ($request->has('low_stock') && $request->low_stock) {
            $query->whereHas('product', function($q) {
                $q->whereRaw('inventory.quantity <= products.min_stock');
            });
        }

        $inventory = $query->orderBy('warehouse_id')
                           ->orderBy('product_id')
                           ->get();

        $totalValue = $inventory->sum(function($item) {
            return $item->quantity * $item->product->purchase_price;
        });

        $totalItems = $inventory->sum('quantity');

        return response()->json([
            'summary' => [
                'total_value' => $totalValue,
                'total_items' => $totalItems,
                'total_products' => $inventory->count(),
            ],
            'inventory' => $inventory,
        ]);
    }

    // Debt Report
    public function debts(Request $request)
    {
        $customerDebts = CustomerDebt::with(['customer', 'salesOrder', 'payments'])
                                     ->where('status', '!=', 'paid')
                                     ->orderBy('due_date')
                                     ->get();

        $supplierDebts = SupplierDebt::with(['supplier', 'purchaseOrder', 'payments'])
                                     ->where('status', '!=', 'paid')
                                     ->orderBy('due_date')
                                     ->get();

        $totalCustomerDebts = $customerDebts->sum('remaining_amount');
        $totalSupplierDebts = $supplierDebts->sum('remaining_amount');

        $overdueCustomerDebts = $customerDebts->filter(function($debt) {
            return $debt->due_date && $debt->due_date < now();
        });

        $overdueSupplierDebts = $supplierDebts->filter(function($debt) {
            return $debt->due_date && $debt->due_date < now();
        });

        return response()->json([
            'summary' => [
                'total_customer_debts' => $totalCustomerDebts,
                'total_supplier_debts' => $totalSupplierDebts,
                'net_debt' => $totalCustomerDebts - $totalSupplierDebts,
                'overdue_customer_debts' => $overdueCustomerDebts->sum('remaining_amount'),
                'overdue_supplier_debts' => $overdueSupplierDebts->sum('remaining_amount'),
            ],
            'customer_debts' => $customerDebts,
            'supplier_debts' => $supplierDebts,
        ]);
    }

    // Profit Report
    public function profit(Request $request)
    {
        $fromDate = $request->from_date ?? now()->startOfMonth()->toDateString();
        $toDate = $request->to_date ?? now()->toDateString();

        // Sales revenue
        $salesRevenue = SalesOrder::whereBetween('order_date', [$fromDate, $toDate])
                                  ->where('status', 'completed')
                                  ->sum('total');

        // Cost of goods sold (from purchase orders)
        $costOfGoods = PurchaseOrder::whereBetween('order_date', [$fromDate, $toDate])
                                    ->where('status', 'completed')
                                    ->sum('total');

        // Calculate gross profit
        $grossProfit = $salesRevenue - $costOfGoods;
        $profitMargin = $salesRevenue > 0 ? ($grossProfit / $salesRevenue) * 100 : 0;

        // Monthly breakdown
        $monthlyProfit = DB::table('sales_orders')
            ->select(
                DB::raw('DATE_FORMAT(order_date, "%Y-%m") as month'),
                DB::raw('SUM(total) as revenue')
            )
            ->whereBetween('order_date', [$fromDate, $toDate])
            ->where('status', 'completed')
            ->groupBy('month')
            ->orderBy('month', 'asc')
            ->get();

        return response()->json([
            'summary' => [
                'total_revenue' => $salesRevenue,
                'cost_of_goods' => $costOfGoods,
                'gross_profit' => $grossProfit,
                'profit_margin' => round($profitMargin, 2),
            ],
            'monthly_profit' => $monthlyProfit,
        ]);
    }
}
