<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Supplier;
use Illuminate\Http\Request;

class SupplierController extends Controller
{
    public function index(Request $request)
    {
        $query = Supplier::withCount(['purchaseOrders', 'debts']);

        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('phone', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            });
        }

        if ($request->has('has_debt') && $request->has_debt) {
            $query->where('total_debt', '>', 0);
        }

        $suppliers = $query->orderBy('created_at', 'desc')
                           ->paginate($request->per_page ?? 20);

        return response()->json($suppliers);
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'nullable|string|max:20',
            'email' => 'nullable|email|unique:suppliers,email',
            'address' => 'nullable|string',
        ]);

        $supplier = Supplier::create($request->all());

        return response()->json([
            'message' => 'Thêm nhà cung cấp thành công',
            'supplier' => $supplier
        ], 201);
    }

    public function show($id)
    {
        $supplier = Supplier::with(['purchaseOrders.items.product', 'debts'])
                            ->findOrFail($id);

        return response()->json($supplier);
    }

    public function update(Request $request, $id)
    {
        $supplier = Supplier::findOrFail($id);

        $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'nullable|string|max:20',
            'email' => 'nullable|email|unique:suppliers,email,' . $id,
            'address' => 'nullable|string',
        ]);

        $supplier->update($request->all());

        return response()->json([
            'message' => 'Cập nhật nhà cung cấp thành công',
            'supplier' => $supplier
        ]);
    }

    public function destroy($id)
    {
        $supplier = Supplier::findOrFail($id);

        if ($supplier->total_debt > 0) {
            return response()->json([
                'message' => 'Không thể xóa nhà cung cấp còn nợ'
            ], 422);
        }

        $supplier->delete();

        return response()->json([
            'message' => 'Xóa nhà cung cấp thành công'
        ]);
    }

    public function debts($id)
    {
        $supplier = Supplier::findOrFail($id);
        $debts = $supplier->debts()
                         ->with(['purchaseOrder', 'payments'])
                         ->orderBy('created_at', 'desc')
                         ->get();

        return response()->json($debts);
    }
}
