<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;

class StoreSalesOrderRequest extends FormRequest
{
    public function authorize(): bool
    {
        return true;
    }

    public function rules(): array
    {
        return [
            'customer_id' => 'required|exists:customers,id',
            'warehouse_id' => 'required|exists:warehouses,id',
            'order_date' => 'required|date',
            'subtotal' => 'required|numeric|min:0',
            'discount' => 'nullable|numeric|min:0',
            'total' => 'required|numeric|min:0',
            'paid_amount' => 'required|numeric|min:0',
            'payment_method' => 'required|in:cash,transfer,card',
            'status' => 'required|in:pending,completed,cancelled',
            'notes' => 'nullable|string',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.price' => 'required|numeric|min:0',
            'items.*.subtotal' => 'required|numeric|min:0',
        ];
    }

    public function withValidator($validator)
    {
        $validator->after(function ($validator) {
            $warehouseId = $this->input('warehouse_id');
            $items = $this->input('items', []);

            foreach ($items as $index => $item) {
                if (isset($item['product_id']) && isset($item['quantity'])) {
                    $product = \App\Models\Product::find($item['product_id']);
                    if ($product) {
                        $availableStock = $product->inventory()
                            ->where('warehouse_id', $warehouseId)
                            ->sum('quantity');
                        
                        if ($item['quantity'] > $availableStock) {
                            $validator->errors()->add(
                                "items.{$index}.quantity",
                                "Không đủ hàng. Tồn kho: {$availableStock}"
                            );
                        }
                    }
                }
            }
        });
    }

    public function messages(): array
    {
        return [
            'customer_id.required' => 'Khách hàng không được để trống',
            'warehouse_id.required' => 'Kho hàng không được để trống',
            'items.required' => 'Phải có ít nhất 1 sản phẩm',
            'items.*.quantity.min' => 'Số lượng phải lớn hơn 0',
        ];
    }
}
