<?php

namespace App\Observers;

use App\Events\SalesOrderCreated;
use App\Models\SalesOrder;
use App\Models\CustomerDebt;

class SalesOrderObserver
{
    public function creating(SalesOrder $order): void
    {
        // Auto-generate order number
        if (empty($order->order_number)) {
            $order->order_number = 'SO-' . date('Ymd') . '-' . str_pad(
                SalesOrder::whereDate('created_at', today())->count() + 1,
                4,
                '0',
                STR_PAD_LEFT
            );
        }
    }

    public function created(SalesOrder $order): void
    {
        // Create debt if not fully paid
        $remainingAmount = $order->total - $order->paid_amount;
        
        if ($remainingAmount > 0) {
            CustomerDebt::create([
                'customer_id' => $order->customer_id,
                'sales_order_id' => $order->id,
                'amount' => $remainingAmount,
                'paid_amount' => 0,
                'remaining_amount' => $remainingAmount,
                'due_date' => now()->addDays(30),
                'status' => 'unpaid',
            ]);
        }

        // Dispatch event
        event(new SalesOrderCreated($order));

        \Log::info("Sales order created: {$order->order_number}");
    }

    public function updated(SalesOrder $order): void
    {
        if ($order->isDirty('status') && $order->status === 'cancelled') {
            \Log::warning("Sales order cancelled: {$order->order_number}");
        }
    }
}
