<?php

namespace App\Repositories;

use App\Models\Product;
use App\Contracts\Repositories\ProductRepositoryInterface;
use Illuminate\Support\Collection;

class ProductRepository extends BaseRepository implements ProductRepositoryInterface
{
    public function __construct(Product $model)
    {
        $this->model = $model;
    }

    public function search(string $keyword): Collection
    {
        return $this->model
            ->where('name', 'like', "%{$keyword}%")
            ->orWhere('sku', 'like', "%{$keyword}%")
            ->with(['category', 'brand', 'unit'])
            ->get();
    }

    public function getByCategory(int $categoryId): Collection
    {
        return $this->model
            ->where('category_id', $categoryId)
            ->with(['category', 'brand', 'unit'])
            ->get();
    }

    public function getByBrand(int $brandId): Collection
    {
        return $this->model
            ->where('brand_id', $brandId)
            ->with(['category', 'brand', 'unit'])
            ->get();
    }

    public function getLowStock(): Collection
    {
        return $this->model
            ->with('inventory')
            ->get()
            ->filter(function ($product) {
                $totalStock = $product->inventory->sum('quantity');
                return $totalStock <= $product->min_stock;
            });
    }

    public function getOutOfStock(): Collection
    {
        return $this->model
            ->with('inventory')
            ->get()
            ->filter(function ($product) {
                return $product->inventory->sum('quantity') === 0;
            });
    }

    public function getBySku(string $sku): ?Product
    {
        return $this->model->where('sku', $sku)->first();
    }
}
