<?php

namespace App\Services;

use App\Models\Inventory;
use App\Models\Product;
use App\Events\LowStockDetected;
use App\Contracts\Services\InventoryServiceInterface;
use Illuminate\Support\Facades\DB;

class InventoryService implements InventoryServiceInterface
{
    public function add(int $productId, int $warehouseId, int $quantity): Inventory
    {
        $inventory = Inventory::firstOrCreate(
            [
                'product_id' => $productId,
                'warehouse_id' => $warehouseId,
            ],
            [
                'quantity' => 0,
            ]
        );

        $inventory->increment('quantity', $quantity);
        $inventory->refresh();

        return $inventory;
    }

    public function deduct(int $productId, int $warehouseId, int $quantity): Inventory
    {
        $inventory = Inventory::where('product_id', $productId)
            ->where('warehouse_id', $warehouseId)
            ->firstOrFail();

        if ($inventory->quantity < $quantity) {
            throw new \Exception('Insufficient stock available');
        }

        $inventory->decrement('quantity', $quantity);
        $inventory->refresh();

        // Check for low stock
        $this->checkLowStock($productId);

        return $inventory;
    }

    public function transfer(
        int $productId,
        int $fromWarehouseId,
        int $toWarehouseId,
        int $quantity
    ): bool {
        return DB::transaction(function () use (
            $productId,
            $fromWarehouseId,
            $toWarehouseId,
            $quantity
        ) {
            $this->deduct($productId, $fromWarehouseId, $quantity);
            $this->add($productId, $toWarehouseId, $quantity);

            return true;
        });
    }

    public function getAvailableStock(int $productId, ?int $warehouseId = null): int
    {
        $query = Inventory::where('product_id', $productId);

        if ($warehouseId) {
            $query->where('warehouse_id', $warehouseId);
        }

        return $query->sum('quantity');
    }

    protected function checkLowStock(int $productId): void
    {
        $product = Product::find($productId);
        $totalStock = $this->getAvailableStock($productId);

        if ($totalStock <= $product->min_stock) {
            event(new LowStockDetected($product, $totalStock));
        }
    }
}
