<?php

namespace App\Services;

use App\Models\SalesOrder;
use App\Models\SalesOrderItem;
use App\Models\Inventory;
use App\Contracts\Services\OrderServiceInterface;
use App\Repositories\OrderRepository;
use Illuminate\Support\Facades\DB;

class OrderService implements OrderServiceInterface
{
    public function __construct(
        protected OrderRepository $orderRepository,
        protected InventoryService $inventoryService
    ) {}

    public function createSalesOrder(array $data): SalesOrder
    {
        return DB::transaction(function () use ($data) {
            // Create order
            $order = $this->orderRepository->create([
                'customer_id' => $data['customer_id'],
                'warehouse_id' => $data['warehouse_id'],
                'user_id' => auth()->id(),
                'order_date' => $data['order_date'],
                'subtotal' => $data['subtotal'] ?? $data['total_amount'] ?? 0,
                'discount' => $data['discount'] ?? 0,
                'total' => $data['total'] ?? $data['total_amount'] ?? 0,
                'paid_amount' => $data['paid_amount'] ?? 0,
                'payment_method' => $data['payment_method'] ?? 'cash',
                'status' => $data['status'] ?? 'pending',
                'notes' => $data['notes'] ?? null,
            ]);

            // Create order items and update inventory
            foreach ($data['items'] as $item) {
                SalesOrderItem::create([
                    'sales_order_id' => $order->id,
                    'product_id' => $item['product_id'],
                    'quantity' => $item['quantity'],
                    'price' => $item['price'],
                    'subtotal' => $item['subtotal'],
                ]);

                // Deduct from inventory
                $this->inventoryService->deduct(
                    $item['product_id'],
                    $data['warehouse_id'],
                    $item['quantity']
                );
            }

            return $order->load('items.product', 'customer');
        });
    }

    public function cancelSalesOrder(int $orderId): bool
    {
        return DB::transaction(function () use ($orderId) {
            $order = $this->orderRepository->findOrFail($orderId);

            if ($order->status === 'cancelled') {
                throw new \Exception('Order is already cancelled');
            }

            // Return inventory
            foreach ($order->items as $item) {
                $this->inventoryService->add(
                    $item->product_id,
                    $order->warehouse_id,
                    $item->quantity
                );
            }

            // Update order status
            $order->update(['status' => 'cancelled']);

            return true;
        });
    }

    public function calculateProfit(SalesOrder $order): float
    {
        $totalCost = $order->items->sum(function ($item) {
            return $item->product->cost_price * $item->quantity;
        });

        return $order->total - $totalCost;
    }
}
