<?php

namespace App\Services;

use App\Models\SalesOrder;
use App\Models\PurchaseOrder;
use App\Models\Product;
use App\Contracts\Services\ReportServiceInterface;
use Illuminate\Support\Facades\DB;

class ReportService implements ReportServiceInterface
{
    public function getSalesReport(string $startDate, string $endDate): array
    {
        $sales = SalesOrder::whereBetween('order_date', [$startDate, $endDate])
            ->where('status', 'completed')
            ->with('items.product')
            ->get();

        $totalRevenue = $sales->sum('total');
        $totalOrders = $sales->count();
        $averageOrderValue = $totalOrders > 0 ? $totalRevenue / $totalOrders : 0;

        // Top selling products
        $topProducts = DB::table('sales_order_items')
            ->join('sales_orders', 'sales_order_items.sales_order_id', '=', 'sales_orders.id')
            ->join('products', 'sales_order_items.product_id', '=', 'products.id')
            ->whereBetween('sales_orders.order_date', [$startDate, $endDate])
            ->where('sales_orders.status', 'completed')
            ->select(
                'products.id',
                'products.name',
                DB::raw('SUM(sales_order_items.quantity) as total_quantity'),
                DB::raw('SUM(sales_order_items.subtotal) as total_revenue')
            )
            ->groupBy('products.id', 'products.name')
            ->orderByDesc('total_quantity')
            ->limit(10)
            ->get();

        return [
            'period' => ['start' => $startDate, 'end' => $endDate],
            'total_revenue' => $totalRevenue,
            'total_orders' => $totalOrders,
            'average_order_value' => $averageOrderValue,
            'top_products' => $topProducts,
        ];
    }

    public function getProfitReport(string $startDate, string $endDate): array
    {
        $sales = SalesOrder::whereBetween('order_date', [$startDate, $endDate])
            ->where('status', 'completed')
            ->with('items.product')
            ->get();

        $purchases = PurchaseOrder::whereBetween('order_date', [$startDate, $endDate])
            ->where('status', 'completed')
            ->get();

        $totalRevenue = $sales->sum('total');
        $totalCost = $purchases->sum('total');
        $grossProfit = $totalRevenue - $totalCost;
        $profitMargin = $totalRevenue > 0 ? ($grossProfit / $totalRevenue) * 100 : 0;

        return [
            'period' => ['start' => $startDate, 'end' => $endDate],
            'total_revenue' => $totalRevenue,
            'total_cost' => $totalCost,
            'gross_profit' => $grossProfit,
            'profit_margin' => round($profitMargin, 2),
        ];
    }

    public function getInventoryReport(): array
    {
        $products = Product::with('inventory')->get();

        $lowStockProducts = $products->filter(function ($product) {
            $totalStock = $product->inventory->sum('quantity');
            return $totalStock <= $product->min_stock;
        });

        $outOfStockProducts = $products->filter(function ($product) {
            return $product->inventory->sum('quantity') === 0;
        });

        $totalInventoryValue = $products->sum(function ($product) {
            $totalStock = $product->inventory->sum('quantity');
            return $totalStock * $product->cost_price;
        });

        return [
            'total_products' => $products->count(),
            'low_stock_count' => $lowStockProducts->count(),
            'out_of_stock_count' => $outOfStockProducts->count(),
            'total_inventory_value' => $totalInventoryValue,
            'low_stock_products' => $lowStockProducts->values(),
            'out_of_stock_products' => $outOfStockProducts->values(),
        ];
    }
}
