<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    /**
     * Run the migrations.
     *
     * FIX CRITICAL BUG #4: Change cascadeOnDelete to restrictOnDelete
     * to prevent accidental data loss
     */
    public function up(): void
    {
        // Disable foreign key checks temporarily
        DB::statement('SET FOREIGN_KEY_CHECKS=0');

        // Fix sales_orders foreign keys
        Schema::table('sales_orders', function (Blueprint $table) {
            // Drop old constraints
            $table->dropForeign(['customer_id']);
            $table->dropForeign(['warehouse_id']);
            $table->dropForeign(['user_id']);
            
            // Add new constraints with restrictOnDelete
            $table->foreign('customer_id')
                  ->references('id')->on('customers')
                  ->restrictOnDelete(); // Cannot delete customer if has orders
            
            $table->foreign('warehouse_id')
                  ->references('id')->on('warehouses')
                  ->restrictOnDelete(); // Cannot delete warehouse if has orders
            
            $table->foreign('user_id')
                  ->references('id')->on('users')
                  ->nullOnDelete(); // Set NULL if user deleted
        });

        // Fix purchase_orders foreign keys
        Schema::table('purchase_orders', function (Blueprint $table) {
            $table->dropForeign(['supplier_id']);
            $table->dropForeign(['warehouse_id']);
            $table->dropForeign(['user_id']);
            
            $table->foreign('supplier_id')
                  ->references('id')->on('suppliers')
                  ->restrictOnDelete();
            
            $table->foreign('warehouse_id')
                  ->references('id')->on('warehouses')
                  ->restrictOnDelete();
            
            $table->foreign('user_id')
                  ->references('id')->on('users')
                  ->nullOnDelete();
        });

        // Fix sales_order_items foreign keys
        Schema::table('sales_order_items', function (Blueprint $table) {
            $table->dropForeign(['product_id']);
            
            $table->foreign('product_id')
                  ->references('id')->on('products')
                  ->restrictOnDelete(); // Cannot delete product if in orders
        });

        // Fix purchase_order_items foreign keys
        Schema::table('purchase_order_items', function (Blueprint $table) {
            $table->dropForeign(['product_id']);
            
            $table->foreign('product_id')
                  ->references('id')->on('products')
                  ->restrictOnDelete();
        });

        // Fix inventory foreign keys
        Schema::table('inventory', function (Blueprint $table) {
            $table->dropForeign(['product_id']);
            $table->dropForeign(['warehouse_id']);
            
            $table->foreign('product_id')
                  ->references('id')->on('products')
                  ->restrictOnDelete(); // Cannot delete product if has inventory
            
            $table->foreign('warehouse_id')
                  ->references('id')->on('warehouses')
                  ->restrictOnDelete();
        });

        // Fix products foreign keys - these can stay cascade or change to nullOnDelete
        Schema::table('products', function (Blueprint $table) {
            $table->dropForeign(['category_id']);
            $table->dropForeign(['brand_id']);
            $table->dropForeign(['unit_id']);
            
            // Option 1: restrictOnDelete (safer)
            $table->foreign('category_id')
                  ->references('id')->on('categories')
                  ->restrictOnDelete();
            
            $table->foreign('brand_id')
                  ->references('id')->on('brands')
                  ->restrictOnDelete();
            
            $table->foreign('unit_id')
                  ->references('id')->on('units')
                  ->restrictOnDelete();
        });

        // Re-enable foreign key checks
        DB::statement('SET FOREIGN_KEY_CHECKS=1');
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        DB::statement('SET FOREIGN_KEY_CHECKS=0');

        // Revert to original cascadeOnDelete
        Schema::table('sales_orders', function (Blueprint $table) {
            $table->dropForeign(['customer_id']);
            $table->dropForeign(['warehouse_id']);
            $table->dropForeign(['user_id']);
            
            $table->foreign('customer_id')->references('id')->on('customers')->cascadeOnDelete();
            $table->foreign('warehouse_id')->references('id')->on('warehouses')->cascadeOnDelete();
            $table->foreign('user_id')->references('id')->on('users')->cascadeOnDelete();
        });

        Schema::table('purchase_orders', function (Blueprint $table) {
            $table->dropForeign(['supplier_id']);
            $table->dropForeign(['warehouse_id']);
            $table->dropForeign(['user_id']);
            
            $table->foreign('supplier_id')->references('id')->on('suppliers')->cascadeOnDelete();
            $table->foreign('warehouse_id')->references('id')->on('warehouses')->cascadeOnDelete();
            $table->foreign('user_id')->references('id')->on('users')->cascadeOnDelete();
        });

        Schema::table('sales_order_items', function (Blueprint $table) {
            $table->dropForeign(['product_id']);
            $table->foreign('product_id')->references('id')->on('products')->cascadeOnDelete();
        });

        Schema::table('purchase_order_items', function (Blueprint $table) {
            $table->dropForeign(['product_id']);
            $table->foreign('product_id')->references('id')->on('products')->cascadeOnDelete();
        });

        Schema::table('inventory', function (Blueprint $table) {
            $table->dropForeign(['product_id']);
            $table->dropForeign(['warehouse_id']);
            
            $table->foreign('product_id')->references('id')->on('products')->cascadeOnDelete();
            $table->foreign('warehouse_id')->references('id')->on('warehouses')->cascadeOnDelete();
        });

        Schema::table('products', function (Blueprint $table) {
            $table->dropForeign(['category_id']);
            $table->dropForeign(['brand_id']);
            $table->dropForeign(['unit_id']);
            
            $table->foreign('category_id')->references('id')->on('categories')->cascadeOnDelete();
            $table->foreign('brand_id')->references('id')->on('brands')->cascadeOnDelete();
            $table->foreign('unit_id')->references('id')->on('units')->cascadeOnDelete();
        });

        DB::statement('SET FOREIGN_KEY_CHECKS=1');
    }
};
