const API_URL = '/api';

// Check authentication
function checkAuth() {
    const token = localStorage.getItem('token');
    if (!token && !window.location.pathname.includes('login.html')) {
        window.location.href = 'login.html';
        return false;
    }
    if (token && window.location.pathname.includes('login.html')) {
        window.location.href = 'dashboard.html';
        return false;
    }
    return true;
}

// Fetch API with authentication
async function fetchAPI(endpoint, options = {}) {
    const token = localStorage.getItem('token');
    
    const defaultOptions = {
        headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'Authorization': `Bearer ${token}`
        }
    };
    
    const finalOptions = {
        ...defaultOptions,
        ...options,
        headers: {
            ...defaultOptions.headers,
            ...options.headers
        }
    };
    
    try {
        const response = await fetch(API_URL + endpoint, finalOptions);
        
        if (response.status === 401) {
            localStorage.clear();
            window.location.href = 'login.html';
            throw new Error('Unauthorized');
        }
        
        const data = await response.json();
        
        if (!response.ok) {
            throw new Error(data.message || 'API Error');
        }
        
        return data;
    } catch (error) {
        console.error('API Error:', error);
        throw error;
    }
}

// Format money
function formatMoney(amount) {
    if (!amount) return '0 đ';
    return new Intl.NumberFormat('vi-VN', {
        style: 'currency',
        currency: 'VND'
    }).format(amount);
}

// Format date
function formatDate(dateString) {
    if (!dateString) return '';
    const date = new Date(dateString);
    return date.toLocaleDateString('vi-VN');
}

// Load sidebar
function loadSidebar() {
    const sidebar = document.getElementById('sidebar');
    if (!sidebar) return;
    
    const currentPage = window.location.pathname.split('/').pop();
    
    sidebar.innerHTML = `
        <div class="sidebar-header">
            <h2>🔧 TrungDuong</h2>
            <p>Quản Lý Bán Hàng</p>
        </div>
        <ul class="sidebar-menu">
            <li><a href="dashboard.html" class="${currentPage === 'dashboard.html' ? 'active' : ''}">📊 Dashboard</a></li>
            <li><a href="products.html" class="${currentPage === 'products.html' ? 'active' : ''}">📦 Sản Phẩm</a></li>
            <li><a href="customers.html" class="${currentPage === 'customers.html' ? 'active' : ''}">👥 Khách Hàng</a></li>
            <li><a href="suppliers.html" class="${currentPage === 'suppliers.html' ? 'active' : ''}">🏭 Nhà Cung Cấp</a></li>
            <li><a href="sales.html" class="${currentPage === 'sales.html' ? 'active' : ''}">💰 Bán Hàng</a></li>
            <li><a href="purchase.html" class="${currentPage === 'purchase.html' ? 'active' : ''}">🛒 Mua Hàng</a></li>
            <li><a href="debts.html" class="${currentPage === 'debts.html' ? 'active' : ''}">💳 Công Nợ</a></li>
            <li><a href="reports.html" class="${currentPage === 'reports.html' ? 'active' : ''}">📈 Báo Cáo</a></li>
        </ul>
        <button class="logout-btn" onclick="logout()">🚪 Đăng Xuất</button>
    `;
}

// Load user info
function loadUserInfo() {
    const userInfo = document.getElementById('userInfo');
    if (!userInfo) return;
    
    const user = JSON.parse(localStorage.getItem('user') || '{}');
    userInfo.innerHTML = `
        <span>👤 ${user.name || 'User'}</span>
        <span class="badge badge-info">${user.role || 'user'}</span>
    `;
}

// Logout
async function logout() {
    try {
        await fetchAPI('/logout', { method: 'POST' });
    } catch (error) {
        console.error('Logout error:', error);
    } finally {
        localStorage.clear();
        window.location.href = 'login.html';
    }
}

// Show notification
function showNotification(message, type = 'success') {
    const notification = document.createElement('div');
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        padding: 15px 25px;
        background: ${type === 'success' ? '#28a745' : '#dc3545'};
        color: white;
        border-radius: 8px;
        box-shadow: 0 4px 12px rgba(0,0,0,0.2);
        z-index: 10000;
        animation: slideIn 0.3s ease;
    `;
    notification.textContent = message;
    
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.style.animation = 'slideOut 0.3s ease';
        setTimeout(() => notification.remove(), 300);
    }, 3000);
}

// Add CSS animations
const style = document.createElement('style');
style.textContent = `
    @keyframes slideIn {
        from {
            transform: translateX(400px);
            opacity: 0;
        }
        to {
            transform: translateX(0);
            opacity: 1;
        }
    }
    
    @keyframes slideOut {
        from {
            transform: translateX(0);
            opacity: 1;
        }
        to {
            transform: translateX(400px);
            opacity: 0;
        }
    }
`;
document.head.appendChild(style);

// Confirm dialog
function confirmDialog(message) {
    return confirm(message);
}

// Initialize page
document.addEventListener('DOMContentLoaded', () => {
    checkAuth();
});
