<?php

namespace Tests\Feature;

use App\Models\User;
use App\Models\Product;
use App\Models\Category;
use App\Models\Brand;
use App\Models\Unit;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class ProductTest extends TestCase
{
    use RefreshDatabase;

    protected function authenticate(): string
    {
        $user = User::factory()->create(['role' => 'admin']);
        return $user->createToken('test-token')->plainTextToken;
    }

    public function test_can_list_products(): void
    {
        $token = $this->authenticate();
        Product::factory()->count(5)->create();

        $response = $this->withHeader('Authorization', 'Bearer ' . $token)
            ->getJson('/api/products');

        $response->assertStatus(200)
            ->assertJsonCount(5, 'data');
    }

    public function test_can_create_product(): void
    {
        $token = $this->authenticate();
        $category = Category::factory()->create();
        $brand = Brand::factory()->create();
        $unit = Unit::factory()->create();

        $productData = [
            'name' => 'Test Product',
            'sku' => 'PRD-TEST-001',
            'category_id' => $category->id,
            'brand_id' => $brand->id,
            'unit_id' => $unit->id,
            'cost_price' => 100000,
            'selling_price' => 150000,
            'min_stock' => 10,
            'description' => 'Test description',
        ];

        $response = $this->withHeader('Authorization', 'Bearer ' . $token)
            ->postJson('/api/products', $productData);

        $response->assertStatus(201)
            ->assertJson([
                'data' => [
                    'name' => 'Test Product',
                    'sku' => 'PRD-TEST-001',
                ],
            ]);

        $this->assertDatabaseHas('products', [
            'name' => 'Test Product',
            'sku' => 'PRD-TEST-001',
        ]);
    }

    public function test_can_show_single_product(): void
    {
        $token = $this->authenticate();
        $product = Product::factory()->create();

        $response = $this->withHeader('Authorization', 'Bearer ' . $token)
            ->getJson("/api/products/{$product->id}");

        $response->assertStatus(200)
            ->assertJson([
                'data' => [
                    'id' => $product->id,
                    'name' => $product->name,
                ],
            ]);
    }

    public function test_can_update_product(): void
    {
        $token = $this->authenticate();
        $product = Product::factory()->create();

        $updateData = [
            'name' => 'Updated Product Name',
            'selling_price' => 200000,
        ];

        $response = $this->withHeader('Authorization', 'Bearer ' . $token)
            ->putJson("/api/products/{$product->id}", $updateData);

        $response->assertStatus(200);

        $this->assertDatabaseHas('products', [
            'id' => $product->id,
            'name' => 'Updated Product Name',
            'selling_price' => 200000,
        ]);
    }

    public function test_can_delete_product(): void
    {
        $token = $this->authenticate();
        $product = Product::factory()->create();

        $response = $this->withHeader('Authorization', 'Bearer ' . $token)
            ->deleteJson("/api/products/{$product->id}");

        $response->assertStatus(200);

        $this->assertDatabaseMissing('products', [
            'id' => $product->id,
        ]);
    }

    public function test_product_validation_fails_with_missing_fields(): void
    {
        $token = $this->authenticate();

        $response = $this->withHeader('Authorization', 'Bearer ' . $token)
            ->postJson('/api/products', [
                'name' => 'Test Product',
                // Missing required fields
            ]);

        $response->assertStatus(422)
            ->assertJsonValidationErrors(['sku', 'category_id', 'brand_id', 'unit_id']);
    }
}
