<?php

namespace Tests\Unit;

use App\Models\Product;
use App\Models\Warehouse;
use App\Models\Inventory;
use App\Services\InventoryService;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class InventoryServiceTest extends TestCase
{
    use RefreshDatabase;

    protected InventoryService $inventoryService;

    protected function setUp(): void
    {
        parent::setUp();
        $this->inventoryService = app(InventoryService::class);
    }

    public function test_can_add_stock_to_inventory(): void
    {
        $product = Product::factory()->create();
        $warehouse = Warehouse::factory()->create();

        $inventory = $this->inventoryService->add(
            $product->id,
            $warehouse->id,
            10
        );

        $this->assertEquals(10, $inventory->quantity);
        $this->assertDatabaseHas('inventory', [
            'product_id' => $product->id,
            'warehouse_id' => $warehouse->id,
            'quantity' => 10,
        ]);
    }

    public function test_can_deduct_stock_from_inventory(): void
    {
        $product = Product::factory()->create();
        $warehouse = Warehouse::factory()->create();

        // Add initial stock
        $this->inventoryService->add($product->id, $warehouse->id, 20);

        // Deduct stock
        $inventory = $this->inventoryService->deduct(
            $product->id,
            $warehouse->id,
            5
        );

        $this->assertEquals(15, $inventory->quantity);
    }

    public function test_cannot_deduct_more_than_available_stock(): void
    {
        $this->expectException(\Exception::class);
        $this->expectExceptionMessage('Insufficient stock available');

        $product = Product::factory()->create();
        $warehouse = Warehouse::factory()->create();

        $this->inventoryService->add($product->id, $warehouse->id, 5);
        $this->inventoryService->deduct($product->id, $warehouse->id, 10);
    }

    public function test_can_transfer_stock_between_warehouses(): void
    {
        $product = Product::factory()->create();
        $warehouse1 = Warehouse::factory()->create();
        $warehouse2 = Warehouse::factory()->create();

        // Add stock to warehouse 1
        $this->inventoryService->add($product->id, $warehouse1->id, 20);

        // Transfer to warehouse 2
        $result = $this->inventoryService->transfer(
            $product->id,
            $warehouse1->id,
            $warehouse2->id,
            10
        );

        $this->assertTrue($result);

        $stock1 = $this->inventoryService->getAvailableStock($product->id, $warehouse1->id);
        $stock2 = $this->inventoryService->getAvailableStock($product->id, $warehouse2->id);

        $this->assertEquals(10, $stock1);
        $this->assertEquals(10, $stock2);
    }

    public function test_can_get_available_stock(): void
    {
        $product = Product::factory()->create();
        $warehouse1 = Warehouse::factory()->create();
        $warehouse2 = Warehouse::factory()->create();

        $this->inventoryService->add($product->id, $warehouse1->id, 15);
        $this->inventoryService->add($product->id, $warehouse2->id, 25);

        $totalStock = $this->inventoryService->getAvailableStock($product->id);
        $this->assertEquals(40, $totalStock);

        $warehouse1Stock = $this->inventoryService->getAvailableStock($product->id, $warehouse1->id);
        $this->assertEquals(15, $warehouse1Stock);
    }
}
